#!/bin/bash
#==============================================================================
#      _          _ _    _____ _    _           
#     / \ _  _ __| (_)__|_   _| |_ (_)_ _  __ _ ®
#    / _ \ || / _` | / _ \| | | ' \| | ' \/ _` |
#   /_/ \_\_,_\__,_|_\___/|_| |_||_|_|_||_\__, |
#                                         |___/ 
#  © 2011-2024 AudioThing® Ltd.             
#
#==============================================================================
# install-standalone.sh
product="JUNE"
ident="JUNE"
version="1.1"

#==============================================================================
# colors
RED='\033[0;31m'
MAINCOLOR='\033[0;36m' # Cyan
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
GREEN='\033[0;32m'
NC='\033[0m' # No Color

#==============================================================================
# change to the script directory
SCRIPTDIR=$(dirname "$0")
cd "$SCRIPTDIR"/ || exit

#==============================================================================
# variables
AS_ROOT=0
QUIET=0
VST2DIR="$HOME/.vst/"
VST3DIR="$HOME/.vst3/"
CLAPDIR="$HOME/.clap/"
STANDALONEDIR="$HOME/.local/bin/"
DESKTOPDIR="$HOME/.local/share/applications"
BASEDIR="${XDG_DATA_HOME:-$HOME/.local/share}/AudioThing"
STANDALONEDIR_SET=0
DESKTOPDIR_SET=0

VST2=y
VST3=y
CLAP=y
STANDALONE=y
PRESETS=y

#==============================================================================
usage() {
    echo -e "By default this installer will install everything in the default paths."
    echo -e "You can customize the installer with these options"

    echo -e "usage: ./install.sh <options>..."
    echo -e "  --as-root          lets you run this script as root - you have to know what you're doing"
    echo -e "  --quiet            agree the license and install without asking questions (default: vst2, vst3, clap, standalone, and presets)"
    echo
    echo -e "  --vst2dir=<path>   specify the vst2 directory. Default: $VST2DIR"
    echo -e "  --vst3dir=<path>   specify the vst3 directory. Default: $VST3DIR"
    echo -e "  --clapdir=<path>   specify the clap directory. Default: $CLAPDIR"
    echo -e "  --standalonedir=<path>   specify the standalone directory. Default: $STANDALONEDIR"
    echo -e "  --desktopdir=<path>      specify the .desktop directory. Default: $DESKTOPDIR"
    echo
    echo -e " in quite mode you can also use the following options:"
    echo -e "  --no-vst2          does not install vst2"
    echo -e "  --no-vst3          does not install vst3"
    echo -e "  --no-clap          does not install clap"
    echo -e "  --no-standalone    does not install standalone"
    echo -e "  --no-presets       does not install presets"
    echo
    exit 0
}

#==============================================================================
# parse arguments
until test $# -eq 0; do
    case "$1" in
    --vst2dir=*)
        VST2DIR="${1/--vst2dir=/}"
        VST2DIR="$(eval echo -e "$VST2DIR")"
        ;;
    --vst3dir=*)
        VST3DIR="${1/--vst3dir=/}"
        VST3DIR="$(eval echo -e "$VST3DIR")"
        ;;
    --clapdir=*)
        CLAPDIR="${1/--clapdir=/}"
        CLAPDIR="$(eval echo -e "$CLAPDIR")"
        ;;
    --standalonedir=*)
        STANDALONEDIR="${1/--standalonedir=/}"
        STANDALONEDIR="$(eval echo -e "$STANDALONEDIR")"
        STANDALONEDIR_SET=1
        ;;
    --desktopdir=*)
        DESKTOPDIR="${1/--desktopdir=/}"
        DESKTOPDIR="$(eval echo -e "$DESKTOPDIR")"
        DESKTOPDIR_SET=1
        ;;
    --no-clap)
        CLAP=n
        ;;
    --no-vst2)
        VST2=n
        ;;
    --no-vst3)
        VST3=n
        ;;
    --no-standalone)
        STANDALONE=n
        ;;
    --no-presets)
        PRESETS=n
        ;;
    --quiet)
        QUIET=1
        VST2=y
        VST3=y
        CLAP=y
        STANDALONE=y
        PRESETS=y
        ;;
    --as-root)
        AS_ROOT=1
        ;;
    --help | -h)
        usage
        ;;
    *)
        echo -e wrong argument "$1"
        usage
        ;;
    esac
    shift
done

#==============================================================================
# standalone and desktop paths
if [ "$AS_ROOT" -eq 1 ]; then
    if [ "$STANDALONEDIR_SET" -eq 0 ]; then
        STANDALONEDIR="/opt/AudioThing"
    fi
    if [ "$DESKTOPDIR_SET" -eq 0 ]; then
        DESKTOPDIR="/usr/share/applications"
    fi
else
    if [ "$STANDALONEDIR_SET" -eq 0 ]; then
        STANDALONEDIR="$HOME/.local/bin/"
    fi
    if [ "$DESKTOPDIR_SET" -eq 0 ]; then
        DESKTOPDIR="$HOME/.local/share/applications"
    fi
fi

#==============================================================================
# prevent root install
if test "$(whoami)" = root -a "$AS_ROOT" -eq 0; then
    echo -e "Please do not install as root."
    exit 1
fi

#==============================================================================
# main
echo -e "--------------------------------------------------------------------------------"
echo -e "    _          _ _    _____ _    _           ";
echo -e "   / \ _  _ __| (_)__|_   _| |_ (_)_ _  __ _ ®";
echo -e "  / _ \ || / _\` | / _ \| | | ' \| | ' \/ _\` |";
echo -e " /_/ \_\_,_\__,_|_\___/|_| |_||_|_|_||_\__, |";
echo -e "                                       |___/ ";
echo -e "--------------------------------------------------------------------------------"
echo -e "${MAINCOLOR}"AudioThing® Linux Installer - "${YELLOW}"$product "$version""${NC}" 
echo -e "--------------------------------------------------------------------------------"

if test $QUIET -eq 0; then
    # installation type
    echo -e "Default installation: ${GREEN}vst2${NC}, ${GREEN}vst3${NC}, ${GREEN}clap${NC}, ${GREEN}standalone${NC}, and ${GREEN}presets${NC}."
    echo -e -n "Do you want to ${YELLOW}customize${NC} the installation? [y/n] "
    read customize

    if test "$customize" = "y"; then
        # options
        echo -e --------------------------------------------------------------------------------
        echo -e -n  "Do you want to install the ${GREEN}vst2${NC} plugin? [y/n] "
        read VST2
        echo -e -n  "Do you want to install the ${GREEN}vst3${NC} plugin? [y/n] "
        read VST3
        echo -e -n  "Do you want to install the ${GREEN}clap${NC} plugin? [y/n] "
        read CLAP
        echo -e -n  "Do you want to install the ${GREEN}standalone${NC} version? [y/n] "
        read STANDALONE
        echo -e -n  "Do you want to install the ${GREEN}presets${NC}? [y/n] "
        read PRESETS
        echo
    fi

    # EULA
    if test -r "${SCRIPTDIR}/Docs/license.txt"; then
        echo -e --------------------------------------------------------------------------------
        cat "${SCRIPTDIR}/Docs/license.txt"
        echo
        echo -e --------------------------------------------------------------------------------
        echo
        while true; do
            echo -e -n "Do you agree the license's terms? [y/n] "
            read answer
            case "$answer" in
            y | Y | yes | YES | Yes) break ;;
            n | N | no | NO | No) exit 1;;
            esac
        done
        echo
    fi
fi

# creates default folder(s)
mkdir -p "$BASEDIR/Licenses/"

# install vst2
if [ "$VST2" = "y" ]; then
    mkdir -p "$VST2DIR"
    cp "${SCRIPTDIR}/Plugins/${ident}.so" "$VST2DIR"
    echo -e The "${GREEN}"vst2"${NC}" has been installed to "${YELLOW}""$VST2DIR""${NC}"
fi

# install vst3
if [ "$VST3" = "y" ]; then
    mkdir -p "$VST3DIR"
    cp -R -H "${SCRIPTDIR}/Plugins/${ident}.vst3" "$VST3DIR"
    echo -e The "${GREEN}"vst3"${NC}" has been installed to "${YELLOW}""$VST3DIR""${NC}"
fi

# install clap
if [ "$CLAP" = "y" ]; then
    mkdir -p "$CLAPDIR"
    cp -R -H "${SCRIPTDIR}/Plugins/${ident}.clap" "$CLAPDIR"
    echo -e The "${GREEN}"clap"${NC}" has been installed to "${YELLOW}""$CLAPDIR""${NC}"
fi

# install standalone
if [ "$STANDALONE" = "y" ]; then
    if [ "$AS_ROOT" -eq 1 ]; then
        sudo mkdir -p "$STANDALONEDIR"
        sudo cp -R -H "${SCRIPTDIR}/Plugins/${ident}" "$STANDALONEDIR"
        sudo cp "${SCRIPTDIR}/Plugins/${ident}.png" "$STANDALONEDIR"
        sudo chmod 777 "${STANDALONEDIR}/${ident}"
        sudo chmod 777 "${STANDALONEDIR}/${ident}.png"
        sudo cp "${SCRIPTDIR}/Plugins/${ident}.desktop" "$DESKTOPDIR"

        # Update the Exec and Icon paths
        sudo sed -i "s|^Exec=.*|Exec=$(echo "${STANDALONEDIR}/${ident}" | sed 's:/\+:/:g')|g" "${DESKTOPDIR}/${ident}.desktop"
        sudo sed -i "s|^Icon=.*|Icon=$(echo "${STANDALONEDIR}/${ident}.png" | sed 's:/\+:/:g')|g" "${DESKTOPDIR}/${ident}.desktop"

        sudo chmod +x "${DESKTOPDIR}/${ident}.desktop"
    else
        mkdir -p "$STANDALONEDIR"
        cp -R -H "${SCRIPTDIR}/Plugins/${ident}" "$STANDALONEDIR"
        cp "${SCRIPTDIR}/Plugins/${ident}.png" "$STANDALONEDIR"
        cp "${SCRIPTDIR}/Plugins/${ident}.desktop" "$DESKTOPDIR"

        # udate the Exec and Icon paths
        sed -i "s|^Exec=.*|Exec=$(echo "${STANDALONEDIR}/${ident}" | sed 's:/\+:/:g')|g" "${DESKTOPDIR}/${ident}.desktop"
        sed -i "s|^Icon=.*|Icon=$(echo "${STANDALONEDIR}/${ident}.png" | sed 's:/\+:/:g')|g" "${DESKTOPDIR}/${ident}.desktop"
        
        gio set "${DESKTOPDIR}/${ident}.desktop" metadata::trusted true
        chmod a+x "${DESKTOPDIR}/${ident}.desktop"
    fi

    echo -e The "${GREEN}"standalone"${NC}" has been installed to "${YELLOW}""$STANDALONEDIR""${NC}"
    echo -e The "${GREEN}".desktop"${NC}" has been installed to "${YELLOW}""$DESKTOPDIR""${NC}"
fi

# install presets
if [ "$PRESETS" = "y" ]; then
    mkdir -p "$BASEDIR/Presets/"
    cp -R -H "$SCRIPTDIR/Presets/$ident" "$BASEDIR/Presets/"
    echo -e The "${GREEN}"presets"${NC}" have been installed to "${YELLOW}""$BASEDIR/Presets/""${NC}"
fi

echo
echo -e "We would like to remind you that we are not able to provide the same level of" 
echo -e "support for these products as we do for the macOS and Windows versions."

